
#include <stdio.h>
#include "CalculateTwiddleFactors.h"

#define SCALE           1L<<15                  /* 2^15 */
#define RANGE           1L<<16                  /* 2^16 */

/*...........................................................................*/

/* Float2Fract implementation. */

fractional Float2Fract (                /* Converts float into fractional */
   float aVal                           /* float value in range [-1, 1) */
) {

   /* Local declarations. */
   long int scale = SCALE;
   long int fractVal = 0.0;
   double decimalVal = 0.0;
   double dummy = 0.0;
   int isOdd = 0;

   /* Convert with convergent rounding and saturation. */
   decimalVal = aVal*scale;
   if (aVal >= 0) {
      fractVal = floor (decimalVal);
      dummy = fractVal/2.0;
      isOdd = (int) ((dummy - floor (dummy))*2.0);
      dummy = decimalVal -fractVal;
      if ((dummy > 0.5) || ((dummy == 0.5) && isOdd)) {
         fractVal += 1.0;
      }
      if (fractVal >= scale) {
         fractVal = scale - 1.0;
      }
   } else {     /* aVal < 0 */
      fractVal = ceil (decimalVal);
      if (fractVal != decimalVal) {
         fractVal -= 1.0;
      }
      dummy = fractVal/2.0;
      isOdd = (int) ((dummy - floor (dummy))*2.0);
      dummy = decimalVal -fractVal;
      if ((dummy > 0.5) || ((dummy == 0.5) && isOdd)) {
         fractVal += 1.0;
      }
      if (fractVal < -scale) {
         fractVal = -scale;
      }
   }

   /* Return fractional value. */
   return ((fractional) fractVal);

} /* end of MatrixInverse */


extern fractcomplex* TwidFactorInit (   /* Initialize twiddle factors */
                                        /* computed in floating point */
                                        /* converted to fractionals */
   int log2N,                           /* log2(N), N is number of factors */
                                        /* NOTE: only half of twiddle factors */
                                        /* are used for I/FFT computation */
   fractcomplex* twidFactors,           /* ptr to twiddle factors */
   int conjFlag                         /* indicates whether to generate */
                                        /* complex conjugates of twiddles */
                                        /* 0 : no conjugates (default) */
                                        /* 1 : conjugates */
                                        /* twidFactors returned */
                                        /* only the first half: */
                                        /* WN(0)...WN(N/2-1) */
                                        /* (or their conjugates) */
) {

   /* Local declarations. */
   fractcomplex* retFactors = twidFactors;
   /* NOTE: instead of wReal, wImag, wRecReal and wRecImag declared     */
   /* as double, it would have been cleaner to declare w and wRec as    */
   /* fractcomplex. In this case, the values w.real, w.imag wRec.real   */
   /* and wRec.imag would have replaced the instances of wReal, wImag,  */
   /* wRecReal and wRecImag, respectively. However, if declared as      */
   /* fractcomplex structures, the PIC30 simulator fails to compute     */
   /* the twiddle factors even though the results are correct when      */
   /* compiling solely under CYGWIN!!! Hence, the variables have        */
   /* been declared individual doubles.                                 */
   double wReal = 0.0;
   double wImag = 0.0;
   double wRecReal = 0.0;
   double wRecImag = 0.0;
   double wTmp = 0.0;
   double arg = PI;                             /* sin/cos argument */
                                                /* (default PI) */
   int numFactors = (1<<log2N)/2;               /* number twiddle factors */
   int cntr = 0;

   /* Trivial first factor value. */
   twidFactors->real = Float2Fract (1.0);
   twidFactors++->imag = 0.0;

   /* Set up twiddle factor computation. */
   arg /= (double) (numFactors);
   wRecReal = wReal = cos(arg);
   /* Twiddle factor conjugates? */
   if (conjFlag) {
      /* Yes. */
      wRecImag = wImag = sin(arg);
   } else {
      /* No. */
      wRecImag = wImag =-sin(arg);
   }

   /* Compute twiddle factors recursively. */
   for (cntr = numFactors-2; cntr >= 0; cntr--) {
      twidFactors->real = Float2Fract (wRecReal);
      (twidFactors++)->imag = Float2Fract (wRecImag);
      wTmp = wRecReal*wReal - wRecImag*wImag;
      wRecImag = wRecReal*wImag + wRecImag*wReal;
      wRecReal = wTmp;
   }

   /* Return destination vector pointer. */
   return (retFactors);

} /* end of TwidFactorInit */

extern fractional* HanningInit (	/* Initialize a Hanning window */
					/* computed in floating point */
					/* converted to fractionals */
   int numElems,			/* number elements in window */
   fractional* window			/* ptr to window */
					/* window returned */
) {

   /* Local declarations. */
   fractional* retWindow = window;
   double arg = 2.0*PI/((double) (numElems-1));
   int cntr = 0;

   /* Compute window factors. */
   for (cntr = 0; cntr < numElems; cntr++) {
      *(window++) = Float2Fract (HANN_0 + HANN_1*cos(arg*cntr));
   }

   /* Return window pointer. */
   return (retWindow);

} /* end of HanningInit */

extern fractional* BlackmanInit (	/* Initialize a Blackman window */
					/* computed in floating point */
					/* converted to fractionals */
   int numElems,			/* number elements in window */
   fractional* window			/* ptr to window */
					/* window returned */
) {

   /* Local declarations. */
   fractional* retWindow = window;
   double arg = 2.0*PI/((double) (numElems-1));
   int cntr = 0;

   /* Compute window factors. */
   for (cntr = 0; cntr < numElems; cntr++) {
      *(window++) = 
         Float2Fract (BLCK_0 + BLCK_1*cos(arg*cntr) + BLCK_2*cos(2*arg*cntr));
   }

   /* Return window pointer. */
   return (retWindow);

} /* end of BlackmanInit */

#define FFT_SIZE 1024
#define FFT_LOGN 10

int main(void) {
   fractcomplex twidFactors[FFT_SIZE/2];
   fractional window[FFT_SIZE];
   int i;

   TwidFactorInit(FFT_LOGN, twidFactors, 0);
   BlackmanInit(FFT_SIZE, window);
   printf("static const fractcomplex twidFactors[%d] = {\n", FFT_SIZE/2);
   for( i = 0; i < FFT_SIZE/2; ++i ) {
      printf("\t{ %d, %d }%s\n", twidFactors[i].real, twidFactors[i].imag, i == FFT_SIZE/2-1 ? "" : ",");
   }
   printf("};\n");

   printf("static const fractcomplex blackman[%d] = {\n", FFT_SIZE/2);
   for( i = 0; i < FFT_SIZE/2; ++i ) {
      printf("\t%d%s\n", window[i], i == FFT_SIZE/2-1 ? "" : ",");
   }
   printf("};\n");
}
